-- https://ackspace.nl/wiki/SP/DIF_transmitter_project
-- From Hackerspace ACKspace
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity spdif_transmitter is
  port(
    bit_clock : in std_logic; -- 128x Fsample
    reset : in std_logic;
    trigger_arm : in std_logic;
    trigger_pol : in std_logic;
    trigger_chan : in std_logic;
    trigger : out std_logic;
    Flip : OUT std_logic_vector(7 downto 0); 
    BRAM0_addr : OUT std_logic_vector(31 downto 0);
    BRAM0_data : IN std_logic_vector(63 downto 0);
    BRAM1_addr : OUT std_logic_vector(31 downto 0);
    BRAM1_data : IN std_logic_vector(63 downto 0);
    pause : in std_logic;
    mute : in std_logic;
    spdif_lp : out std_logic;
    spdif_hp : out std_logic
  );
end entity spdif_transmitter;

architecture behavioral of spdif_transmitter is

  signal bit_counter : std_logic_vector(5 downto 0) := (others => '0');
  signal frame_counter : std_logic_vector(8 downto 0) := (others => '0');
  signal channel_status_shift : std_logic_vector(23 downto 0);
  signal channel_status : std_logic_vector(23 downto 0) := "001000000000000001000000";
  signal addr : std_logic_vector(18 downto 0) := (others => '0');
  signal channelA : std_logic := '0';

  signal data_in_bufferLP : std_logic_vector(23 downto 0);
  signal data_in_bufferHP : std_logic_vector(23 downto 0);

  signal data_out_bufferLP : std_logic_vector(7 downto 0);
  signal data_out_bufferHP : std_logic_vector(7 downto 0);

  signal data_biphaseLP : std_logic := '0';
  signal data_biphaseHP : std_logic := '0';

  signal parityLP : std_logic;
  signal parityHP : std_logic;

  signal s_trigger : std_logic := '0';
  signal trigger_flipped : std_logic := '0';
  signal trigger_count : std_logic_vector(1 downto 0) := (others => '0');
 
  signal muteCount : integer := 0;
  signal lastMute : std_logic;
  signal lastHPSignA : std_logic;
  signal lastLPSignA : std_logic;
  signal lastHPCountA : integer := 0;
  signal lastLPCountA : integer := 0;
  signal lastHPSignB : std_logic;
  signal lastLPSignB : std_logic;
  signal lastHPCountB : integer := 0;
  signal lastLPCountB : integer := 0;
begin
  BRAM0_addr <= x"0000" & addr(15 downto 0);
  BRAM1_addr <= x"0000" & addr(15 downto 0); 
  Flip <= "00001" & addr(18 downto 16);
  trigger <= s_trigger;
  
  bit_clock_counter : process (bit_clock)
  begin
    if bit_clock'event and bit_clock = '1' then
      if reset = '1' then 
        bit_counter <= (others => '0');
      else
        bit_counter <= bit_counter + 1;
      end if;
    end if;
  end process bit_clock_counter;

  -- when trigger is armed from PS, set trigger for the first non-zero sample
  -- trigger has to be desarmed from PS before enabling the cycle again
  process (bit_clock)
  begin
    if bit_clock'event and bit_clock = '1' then
      if reset = '1' then 
        trigger_flipped <= '0';
        s_trigger <= '0';
        trigger_count <= "00";
      else
        s_trigger <= '0';
        if trigger_arm = '1' then
          if trigger_pol = '1' then
            if data_in_bufferLP /= x"0000" or data_in_bufferHP /= x"0000" then
              if trigger_flipped = '0' then
                trigger_flipped <= '1';
                s_trigger <= '1';
              end if;
            end if;
          else
            if channelA = trigger_chan and bit_counter = "000100" then
              if data_in_bufferLP = x"0000" or data_in_bufferHP = x"0000" then
                if trigger_flipped = '0' then
                  if trigger_count = "10" then
                    trigger_flipped <= '1';
                    s_trigger <= '1';
                  else
                    trigger_count <= trigger_count + 1;
                  end if;
                end if;
              else
                trigger_count <= "00";
              end if;
            end if;
          end if;
        else
          trigger_flipped <= '0';  -- trigger disarmed from PS
        end if;
      end if;
    end if;
  end process;

  data_latch : process (bit_clock)
  variable dataHP : std_logic_vector(15 downto 0);
  variable dataLP : std_logic_vector(15 downto 0);
  variable datHP : std_logic_vector(15 downto 0);
  variable datLP : std_logic_vector(15 downto 0);
  variable datHPS : std_logic_vector(15 downto 0);
  variable datLPS : std_logic_vector(15 downto 0);
  variable count : integer;
  
  begin
    if bit_clock'event and bit_clock = '1' then
      if reset = '1' then 
        frame_counter <= (others => '0');
      else
        parityLP <= data_in_bufferLP(23) xor data_in_bufferLP(22) xor data_in_bufferLP(21) xor data_in_bufferLP(20) xor data_in_bufferLP(19) xor data_in_bufferLP(18) xor data_in_bufferLP(17)  xor data_in_bufferLP(16) xor data_in_bufferLP(15) xor data_in_bufferLP(14) xor data_in_bufferLP(13) xor data_in_bufferLP(12) xor data_in_bufferLP(11) xor data_in_bufferLP(10) xor data_in_bufferLP(9) xor data_in_bufferLP(8) xor data_in_bufferLP(7) xor data_in_bufferLP(6) xor data_in_bufferLP(5) xor data_in_bufferLP(4) xor data_in_bufferLP(3) xor data_in_bufferLP(2) xor data_in_bufferLP(1) xor data_in_bufferLP(0) xor channel_status_shift(23);

        parityHP <= data_in_bufferHP(23) xor data_in_bufferHP(22) xor data_in_bufferHP(21) xor data_in_bufferHP(20) xor data_in_bufferHP(19) xor data_in_bufferHP(18) xor data_in_bufferHP(17)  xor data_in_bufferHP(16) xor data_in_bufferHP(15) xor data_in_bufferHP(14) xor data_in_bufferHP(13) xor data_in_bufferHP(12) xor data_in_bufferHP(11) xor data_in_bufferHP(10) xor data_in_bufferHP(9) xor data_in_bufferHP(8) xor data_in_bufferHP(7) xor data_in_bufferHP(6) xor data_in_bufferHP(5) xor data_in_bufferHP(4) xor data_in_bufferHP(3) xor data_in_bufferHP(2) xor data_in_bufferHP(1) xor data_in_bufferHP(0) xor channel_status_shift(23);

        if lastMute /= mute then
          muteCount <= 0;
          lastMute <= mute;
          lastHPCountA <= 0;
          lastLPCountA <= 0;
          lastHPCountB <= 0;
          lastLPCountB <= 0;
        end if;
        if bit_counter = "000011" then
          if channelA = '1' then
            if addr(16) = '0' then
              dataHP := BRAM0_data(15 downto 0); 
              dataLP := BRAM0_data(47 downto 32); 
            else
              dataHP := BRAM1_data(15 downto 0); 
              dataLP := BRAM1_data(47 downto 32); 
            end if;
          else
            if addr(16) = '0' then
              dataHP := BRAM0_data(31 downto 16); 
              dataLP := BRAM0_data(63 downto 48); 
            else
              dataHP := BRAM1_data(31 downto 16); 
              dataLP := BRAM1_data(63 downto 48);
            end if;
          end if;
          
          datHPS := x"0000";
          datLPS := x"0000";
          if mute = '1' then
            data_in_bufferHP <= x"0000" & x"00";
            data_in_bufferLP <= x"0000" & x"00";
          else
            data_in_bufferHP <= dataHP & x"00";
            data_in_bufferLP <= dataLP & x"00";
          end if;
          -- 1/8 of sample
          if dataHP(15) = '0' then
            datHP := "000" & dataHP(15 downto 3);
          else
            datHP := x"ffff" - dataHP;
            datHP := "000" & datHP(15 downto 3);
          end if;
          if dataLP(15) = '0' then
            datLP := "000" & dataLP(15 downto 3);
          else
            datLP := x"ffff" - dataLP;
            datLP := "000" & datLP(15 downto 3);
          end if;
          if muteCount < 40000 then
            -- decreasing/increasing in 8 steps of 5000 samples each
            muteCount <= muteCount + 1;    
            count := 1;
            for i in 0 to 7 loop
              if muteCount > count then
                if channelA = '1' then
                  -- high-pass left channel
                  if count <= lastHPCountA then
                    datHPS := datHPS + datHP;
                  else
                    -- wait to cross zero before moving to next step
                    if lastHPSignA /= dataHP(15) then
                      lastHPCountA <= count;
                    end if;
                  end if;
                  -- low-pass left channel     
                  if count <= lastLPCountA then
                    datLPS := datLPS + datLP;  
                  else
                    -- wait to cross zero before moving to next step
                    if lastLPSignA /= dataLP(15) then
                      lastLPCountA <= count;
                    end if;
                  end if;
                  
                else
                  -- high-pass right channel
                  if count <= lastHPCountB then
                    datHPS := datHPS + datHP;
                  else
                    -- wait to cross zero before moving to next step
                    if lastHPSignB /= dataHP(15) then
                      lastHPCountB <= count;
                    end if;
                  end if;
                  -- low-pass right channel  
                  if count <= lastLPCountB then
                    datLPS := datLPS + datLP;
                  else
                    -- wait to cross zero before moving to next step
                    if lastLPSignB /= dataLP(15) then
                      lastLPCountB <= count;
                    end if;
                  end if;
                end if;
              end if;
              count := count + 5000; -- step length
            end loop;
            -- apply the ramp fading value to the sample
            if mute = '1' then
              if dataHP(15) = '0' then
                dataHP := dataHP - datHPS;
              else
                dataHP := dataHP + datHPS;
              end if;
              if dataLP(15) = '0' then
                dataLP := dataLP - datLPS;
              else
                dataLP := dataLP + datLPS;
              end if;
            else
              if dataHP(15) = '0' then
                dataHP := datHPS;
              else
                dataHP := x"ffff" - datHPS;
              end if;
              if dataLP(15) = '0' then
                dataLP := datLPS;
              else
                dataLP := x"ffff" - datLPS;
              end if;         
            end if;
            data_in_bufferHP <= dataHP & x"00";
            data_in_bufferLP <= dataLP & x"00";
          end if;
          if channelA = '1' then
            lastHPSignA <= dataHP(15);
            lastLPSignA <= dataLP(15);
          else
            lastHPSignB <= dataHP(15);
            lastLPSignB <= dataLP(15);
          end if;
        end if;
        if bit_counter = "111111" then
          if frame_counter = "101111111" then
            frame_counter <= (others => '0');
          else
            frame_counter <= frame_counter + 1;
          end if;
        end if;
      end if;
    end if;
  end process data_latch;

  data_output : process (bit_clock)
  begin
    if bit_clock'event and bit_clock = '1' then
      if reset = '1' then 
        channel_status <= "001000000000000001000000";
        data_out_bufferLP <= (others => '0');
        data_out_bufferHP <= (others => '0');
        channelA <= '0';
        addr <= (others => '0');
      else
        if bit_counter = "111111" then
          if frame_counter = "101111111" then -- next frame is 0, load preamble Z (B)
            if pause = '0' then
              addr <= addr+8;  -- next stereo bi-amp sample 
            end if;
            channelA <= '1';
            channel_status_shift <= channel_status;
            data_out_bufferLP <= "10011100";
            data_out_bufferHP <= "10011100";
          else
            if frame_counter(0) = '1' then -- next frame is even, load preamble X (M)
              channel_status_shift <= channel_status_shift(22 downto 0) & '0';
              data_out_bufferLP <= "10010011";
              data_out_bufferHP <= "10010011";
              if pause = '0' then
                addr <= addr+8;  -- next stereo bi-amp sample 
              end if;
              channelA <= '1';
            else -- next frame is odd, load preable Y (W)
              data_out_bufferLP <= "10010110";
              data_out_bufferHP <= "10010110";
              channelA <= '0';
            end if;
          end if;
        else
          if bit_counter(2 downto 0) = "111" then -- load new part of data into buffer
            case bit_counter(5 downto 3) is
            when "000" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(0) & '1' & data_in_bufferLP(1) & '1' & data_in_bufferLP(2) & '1' & data_in_bufferLP(3);
              data_out_bufferHP <= '1' & data_in_bufferHP(0) & '1' & data_in_bufferHP(1) & '1' & data_in_bufferHP(2) & '1' & data_in_bufferHP(3);
            when "001" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(4) & '1' & data_in_bufferLP(5) & '1' & data_in_bufferLP(6) & '1' & data_in_bufferLP(7);
              data_out_bufferHP <= '1' & data_in_bufferHP(4) & '1' & data_in_bufferHP(5) & '1' & data_in_bufferHP(6) & '1' & data_in_bufferHP(7);
            when "010" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(8) & '1' & data_in_bufferLP(9) & '1' & data_in_bufferLP(10) & '1' & data_in_bufferLP(11);
              data_out_bufferHP <= '1' & data_in_bufferHP(8) & '1' & data_in_bufferHP(9) & '1' & data_in_bufferHP(10) & '1' & data_in_bufferHP(11);
            when "011" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(12) & '1' & data_in_bufferLP(13) & '1' & data_in_bufferLP(14) & '1' & data_in_bufferLP(15);
              data_out_bufferHP <= '1' & data_in_bufferHP(12) & '1' & data_in_bufferHP(13) & '1' & data_in_bufferHP(14) & '1' & data_in_bufferHP(15);
            when "100" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(16) & '1' & data_in_bufferLP(17) & '1' & data_in_bufferLP(18) & '1' & data_in_bufferLP(19);
              data_out_bufferHP <= '1' & data_in_bufferHP(16) & '1' & data_in_bufferHP(17) & '1' & data_in_bufferHP(18) & '1' & data_in_bufferHP(19);
            when "101" =>
              data_out_bufferLP <= '1' & data_in_bufferLP(20) & '1' & data_in_bufferLP(21) & '1' & data_in_bufferLP(22) & '1' & data_in_bufferLP(23);
              data_out_bufferHP <= '1' & data_in_bufferHP(20) & '1' & data_in_bufferHP(21) & '1' & data_in_bufferHP(22) & '1' & data_in_bufferHP(23);
            when "110" =>
              data_out_bufferLP <= "10101" & channel_status_shift(23) & "1" & parityLP;
              data_out_bufferHP <= "10101" & channel_status_shift(23) & "1" & parityHP;
            when others =>
            end case;
          else
            data_out_bufferLP <= data_out_bufferLP(6 downto 0) & '0';
            data_out_bufferHP <= data_out_bufferHP(6 downto 0) & '0';
          end if;
        end if;
      end if;
    end if;
  end process data_output;
  
  biphaser : process (bit_clock)
  begin
    if bit_clock'event and bit_clock = '1' then
      if reset = '1' then
        data_biphaseLP <= '0';
        data_biphaseHP <= '0';
      else
        if data_out_bufferLP(data_out_bufferLP'left) = '1' then
          data_biphaseLP <= not data_biphaseLP;
        end if;
        if data_out_bufferHP(data_out_bufferHP'left) = '1' then
          data_biphaseHP <= not data_biphaseHP;
        end if;
      end if;
    end if;
  end process biphaser;
 
  spdif_lp <= data_biphaseLP;
  spdif_hp <= data_biphaseHP;
 
end behavioral;
